import json, base64, time, uuid, requests, os
from pathlib import Path
from typing import Dict, Any, Optional

# ======== Config ========
DOCUMENTS_CLIENT_ID = "<your_client_id_here>"
DOCUMENTS_CLIENT_SECRET = "<your_client_secret_here>"

PDF_SERVICES_CLIENT_ID = "<your_client_id_here>"
PDF_SERVICES_CLIENT_SECRET = "<your_client_secret_here>"

FOXIT_HOST = "https://na1.fusion.foxit.com"  # Foxit API host
HR_ASSIST_URL = "https://misty-hall-739a.jorge-euceda.workers.dev/hr-assist"

BASE_DIR = Path(__file__).parent.resolve()
INPUTS_DIR = BASE_DIR / "inputs"
OUTPUTS_DIR = BASE_DIR / "outputs"
INPUTS_DIR.mkdir(parents=True, exist_ok=True)
OUTPUTS_DIR.mkdir(parents=True, exist_ok=True)

RESUME_PATH     = INPUTS_DIR / "input_resume.pdf"
TEMPLATE_PATH   = INPUTS_DIR / "hr_report_template.docx"
POLICY_PATH     = INPUTS_DIR / "evaluation_policy.json"

OUTPUT_PDF_PATH = OUTPUTS_DIR / "HR_Report.pdf"
EVENT_LOG_PATH  = OUTPUTS_DIR / "events.jsonl"

# ======== Helpers ========
SESSION = requests.Session()
SESSION.headers.update({"User-Agent": "apiworld-2025-resume-demo/1.1"})

def emit_event(event_type: str, payload: Dict[str, Any], trace_id: str):
    evt = {
        "eventId": str(uuid.uuid4()),
        "eventType": event_type,
        "occurredAt": time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime()),
        "traceId": trace_id,
        "payload": payload,
    }
    if not EVENT_LOG_PATH.exists():
        EVENT_LOG_PATH.write_text("", encoding="utf-8")
    with EVENT_LOG_PATH.open("a", encoding="utf-8") as f:
        f.write(json.dumps(evt) + "\n")
    print(f"[event] {event_type}")

def _retry(fn, attempts=6, delay=2.0):
    for i in range(attempts):
        try:
            return fn()
        except requests.RequestException:
            if i == attempts - 1:
                raise
            time.sleep(delay)

def load_policy(trace_id: str) -> Dict[str, Any]:
    """Load evaluation policy from inputs/evaluation_policy.json, or fall back to sane defaults."""
    if POLICY_PATH.exists():
        policy = json.loads(POLICY_PATH.read_text(encoding="utf-8"))
    else:
        policy = {
            "policyId": "EvaluationPolicy#v1.0",
            "job_description": "Looking for a software engineer with expertise in C++, Python, and AWS cloud services. Experience building scalable applications in agile teams; familiarity with DevOps and CI/CD.",
            "overall_summary": "Make the summary as short as possible",
            "hard_requirements": ["C++", "python", "aws"]
        }
    emit_event("PolicyLoaded", {"policy": policy}, trace_id)
    return policy

# ======= Step 1: Extract text from PDF using Foxit PDF Services ========
def extract_pdf_content(pdf_path: Path, trace_id: str) -> str:
    def upload_doc(path: Path) -> Dict[str, Any]:
        headers = {"client_id": PDF_SERVICES_CLIENT_ID, "client_secret": PDF_SERVICES_CLIENT_SECRET}
        files = {"file": (path.name, path.open("rb"))}
        r = _retry(lambda: SESSION.post(f"{FOXIT_HOST}/pdf-services/api/documents/upload",
                                        files=files, headers=headers, timeout=60))
        r.raise_for_status()
        return r.json()

    def extract_pdf(doc_id: str, extract_type: str = "TEXT", page_range: Optional[str] = None) -> Dict[str, Any]:
        headers = {
            "client_id": PDF_SERVICES_CLIENT_ID, "client_secret": PDF_SERVICES_CLIENT_SECRET,
            "Content-Type": "application/json"
        }
        body = {"documentId": doc_id, "extractType": extract_type}
        if page_range:
            body["pageRange"] = page_range
        r = _retry(lambda: SESSION.post(f"{FOXIT_HOST}/pdf-services/api/documents/modify/pdf-extract",
                                        json=body, headers=headers, timeout=60))
        r.raise_for_status()
        return r.json()

    def wait_task(task_id: str) -> Dict[str, Any]:
        headers = {
            "client_id": PDF_SERVICES_CLIENT_ID, "client_secret": PDF_SERVICES_CLIENT_SECRET,
            "Content-Type": "application/json"
        }
        while True:
            r = _retry(lambda: SESSION.get(f"{FOXIT_HOST}/pdf-services/api/tasks/{task_id}",
                                           headers=headers, timeout=30))
            r.raise_for_status()
            status = r.json()
            state = status.get("status")
            print(f"[foxit] task={task_id} status={state} progress={status.get('progress')}")
            if state == "COMPLETED":
                return status
            if state == "FAILED":
                raise RuntimeError(f"PDF extract failed: {status}")
            time.sleep(2)

    def download_result(doc_id: str) -> str:
        headers = {"client_id": PDF_SERVICES_CLIENT_ID, "client_secret": PDF_SERVICES_CLIENT_SECRET}
        r = _retry(lambda: SESSION.get(f"{FOXIT_HOST}/pdf-services/api/documents/{doc_id}/download",
                                       headers=headers, timeout=60))
        r.raise_for_status()
        return r.text

    emit_event("ResumeUploaded", {"path": str(pdf_path)}, trace_id)
    doc = upload_doc(pdf_path)
    emit_event("DocumentStored", {"documentId": doc["documentId"]}, trace_id)

    task = extract_pdf(doc["documentId"])
    emit_event("ExtractionRequested", {"taskId": task["taskId"]}, trace_id)

    result = wait_task(task["taskId"])
    emit_event("ResumeExtracted", {"resultDocumentId": result["resultDocumentId"]}, trace_id)

    text = download_result(result["resultDocumentId"])
    return text

# ======== Step 2: Call HR Assist AI for resume analysis ========
def call_hr_assist(resume_text: str, policy: Dict[str, Any], trace_id: str) -> dict:
    """
    Pass the policy into the AI so changing policy yields a different response.
    """
    headers = {"Content-Type": "application/json"}
    policy_json = json.dumps(policy, ensure_ascii=False)
    system_instructions = (
        "You are an HR Assistant AI. Convert the resume and job description into JSON with keys: "
        "candidate_name, candidate_phone_number, overall_fit_score (0-100), matching_skills (string[]), "
        "skill_gaps (string[]), overall_summary (string).\n\n"
        "Scoring rules:\n"
        "- Use the provided POLICY JSON strictly, it will include a job_description field.\n"
        "- Normalize skills to lowercase tokens and dedupe.\n"
        "- If any hard_requirements are missing, cap overall_fit_score at min(min_score-1, 30).\n"
        "- Otherwise compute overall_fit_score by summing weights for matched skills, "
        "scale to 0-100, and round to nearest integer.\n"
        "- Do not add extra keys. Unknown values must be the string 'Unknown'."
    )

    data = {
        "model": "gpt-5-nano",
        "response_format": {"type": "json_object"},
        "messages": [
            {"role": "system", "content": system_instructions},
            {"role": "user",
             "content":
                 f"POLICY:\n```json\n{policy_json}\n```\n\n"
                 f'RESUME:\n""" {resume_text} """\n\n'}
        ]
    }

    resp = _retry(lambda: SESSION.post(HR_ASSIST_URL, headers=headers, json=data, timeout=120))
    resp.raise_for_status()
    payload = resp.json()
    content = payload["choices"][0]["message"]["content"]

    try:
        report = json.loads(content)
    except json.JSONDecodeError as e:
        raise RuntimeError(f"Model did not return valid JSON: {content}") from e

    emit_event("DecisionProposed", {
        "fitScore": report.get("overall_fit_score"),
        "policy": policy,
        "policyRef": policy.get("policyId")
    }, trace_id)
    return report

# ======== Step 3: Call Foxit Document Generation ========
def foxit_docgen_from_base64(base64_template: str, document_values: dict, trace_id: str) -> bytes:
    url = f"{FOXIT_HOST}/document-generation/api/GenerateDocumentBase64"
    headers = {
        "client_id": DOCUMENTS_CLIENT_ID, "client_secret": DOCUMENTS_CLIENT_SECRET,
        "Content-Type": "application/json",
    }
    body = {"outputFormat": "pdf", "documentValues": document_values, "base64FileString": base64_template}
    r = _retry(lambda: SESSION.post(url, headers=headers, json=body, timeout=120))
    r.raise_for_status()
    data = r.json()
    if not data.get("base64FileString"):
        raise RuntimeError(f"DocGen failed: {data}")
    emit_event("DocumentGenerated", {"template": str(TEMPLATE_PATH.name)}, trace_id)
    return base64.b64decode(data["base64FileString"].encode("ascii"))

# ======== Main flow ========
if __name__ == "__main__":
    trace_id = str(uuid.uuid4())
    emit_event("RunStarted", {"script": "resume_to_report.py"}, trace_id)

    # Load policy (changing this file changes AI outcomes)
    policy = load_policy(trace_id)

    if not RESUME_PATH.exists():
        raise FileNotFoundError(f"Missing resume at {RESUME_PATH}")
    if not TEMPLATE_PATH.exists():
        raise FileNotFoundError(f"Missing template at {TEMPLATE_PATH}")

    resume_text = extract_pdf_content(RESUME_PATH, trace_id)
    print("Extracted resume text. Calling HR Assist…")

    report = call_hr_assist(resume_text, policy, trace_id)
    print("AI Report:", json.dumps(report, indent=2))

    template_b64 = base64.b64encode(TEMPLATE_PATH.read_bytes()).decode("utf-8")
    document_values = {
        "candidate_name":         report.get("candidate_name", "Unknown"),
        "candidate_phone_number": report.get("candidate_phone_number", "Unknown"),
        "overall_fit_score":      report.get("overall_fit_score", 0),
        "matching_skills":        report.get("matching_skills", []),
        "skill_gaps":             report.get("skill_gaps", []),
        "overall_summary":        report.get("overall_summary", ""),
        "trace_id":               trace_id,
        "policy_ref":            policy.get("policyId", "HiringPolicy#v1")
    }

    pdf_bytes = foxit_docgen_from_base64(template_b64, document_values, trace_id)
    OUTPUT_PDF_PATH.write_bytes(pdf_bytes)
    emit_event("Delivered", {"path": str(OUTPUT_PDF_PATH.resolve())}, trace_id)
    emit_event("Archived", {"strategy": "local", "output": str(OUTPUT_PDF_PATH)}, trace_id)

    print(f"Saved report to: {OUTPUT_PDF_PATH.resolve()}")
    print(f"Event log: {EVENT_LOG_PATH.resolve()}")
